package gov.va.med.mhv.core.validate;

import java.math.BigDecimal;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.LinkedHashMap;


public final class Validations {

	private Validations() {}
	
	public static void validateMaximumLength(String fieldName,String value,int maxLength, boolean required, LinkedHashMap<String, String>  validationErrors){
		
		if(required){
			if(!hasValue(value)){
				validationErrors.put(fieldName, fieldName + " is required");
				return;
			}
		}
		
		if(hasValue(value)){
			if(value.length() > maxLength){
				validationErrors.put(fieldName, fieldName + " is too long")	;
			}
		}
		
	}

	public static boolean hasValue(String s){
		return null != s && s.trim().length() > 0;
	}
	
	
	public static void validateRequired(String fieldName, String value, LinkedHashMap<String, String>  validationErrors) {
		if (!hasValue(value)) {
			validationErrors.put(fieldName, fieldName + " is required");
		}
	}


	public static void validateRequired(String fieldName, String month, String day, String year, String property, LinkedHashMap<String, String>  validationErrors) {
		if (!hasValue(day) || !hasValue(month) && !hasValue(year)) {
			validationErrors.put(fieldName, fieldName + " is required");
		}
	}
	
	public static void validateMinValue(String fieldName, String value, BigDecimal minValue, LinkedHashMap<String, String>  validationErrors) {
		if (isValidNumber(value)) {
			validateMinValue(fieldName, new BigDecimal(value), minValue, validationErrors);
		}
	}
	
	public static void validateMinValue(String fieldName, Comparable value, Comparable minValue, LinkedHashMap<String, String>  validationErrors) {
		if (value != null && value.compareTo(minValue) < 0) {
			//TODO: change the error message
			validationErrors.put(fieldName, "Minimum value for " + fieldName + " is not valid");
		}
	}
	
	public static void validateMaxValue(String fieldName, String value, BigDecimal maxValue, LinkedHashMap<String, String>  validationErrors) {
		if (isValidNumber(value)) {
			validateMaxValue(fieldName, new BigDecimal(value), maxValue, validationErrors);
		}
	}
	
	public static void validateMaxValue(String fieldName, Comparable value, Comparable maxValue, LinkedHashMap<String, String>  validationErrors) {
		if (value != null && value.compareTo(maxValue) > 0) {
			//TODO: change the error message
			validationErrors.put(fieldName, "Maximum value for " + fieldName + " is not valid");
		}
	}
	
	public static void validateMinLength(String fieldName, String value, int minLength, LinkedHashMap<String, String>  validationErrors) {
		if (hasValue(value) && value.trim().length() < minLength) {
			//TODO: change the error message
			validationErrors.put(fieldName, "Miminum length not met for " + fieldName);
		}
	}
	
	public static void validateMaxLength(String fieldName, String value, int maxLength, LinkedHashMap<String, String>  validationErrors) {
		if (hasValue(value) && value.trim().length() > maxLength) {
			//TODO: change the error message
			//validationErrors.put(fieldName, "Maximum length not met for " + fieldName);
			validationErrors.put(fieldName, "The " + fieldName + " value cannot be more than "+ maxLength +" characters long");
		}
	}
	
	public static void validateScale(String fieldName, String value, int scale, LinkedHashMap<String, String>  validationErrors) {
		if (isValidNumber(value)) {
			validateScale(fieldName, new BigDecimal(value), scale, validationErrors);
		}
	}
	
	public static void validateScale(String fieldName, BigDecimal value, int scale, LinkedHashMap<String, String>  validationErrors) {
		if (value != null && value.scale() > scale) {
			//TODO: change the error message
			validationErrors.put(fieldName, "Invalid Scale for " + fieldName);
		}
	}
	
	public static void validateEnumeration(String fieldName, Enum value, Enum enumeratedValue, LinkedHashMap<String, String>  validationErrors) {
		if (value != null && !value.equals(enumeratedValue)) {
			//TODO: change the error message
			validationErrors.put(fieldName, "Invalid Enum Instance for " + fieldName);
		}
	}
	//TODO:
	public static void validateType(String fieldName, Object value, String validatorType, LinkedHashMap<String, String>  validationErrors) {
		Validator v = ValidatorFactory.getInstance().getValidator(validatorType);
		validationErrors.put(fieldName, "Invalid Number for " + fieldName);
		//messages.addMessages(v.validate(value, new String [] {property}, new Object [] {label}));
	}
	
	/**
	 * Ensure that a string value can be converted into an integer.
	 * @param value
	 * @param key
	 */
	public static void validateInteger(String fieldName, String value, LinkedHashMap<String, String>  validationErrors) {
		if( hasValue(value) && ! Validations.isValidInteger( value ) ) {
			//messages.addMessage(MessageUtils.createErrorMessage(CoreMessages.INVALID_NUMBER, new String [] {property}, new Object[]{ value }));
			validationErrors.put(fieldName, "Invalid Number for " + fieldName);
		}
	}

	
	/**
	 * Ensure that a string value can be converted into a long.
	 * @param value
	 * @param key
	 */
	public static void validateLong(String fieldName, String value, LinkedHashMap<String, String>  validationErrors) {
		if( hasValue(value) && ! Validations.isValidLong( value ) ) {
			//messages.addMessage(MessageUtils.createErrorMessage(CoreMessages.INVALID_NUMBER, new String [] {property}, new Object[]{ value }));
			validationErrors.put(fieldName, "Invalid Number for " + fieldName);
		}
	}

	public static final void validateBoolean(String fieldName, String value, LinkedHashMap<String, String>  validationErrors) {
		// TODO Implement this.  For now, can leave blank, as unknown strings will become false Booleans, which is fine
	}
	
	/**
	 * Ensure that a string value can be converted into a decimal.
	 * @param value
	 * @param key
	 */
	public static void validateBigDecimal(String fieldName, String value, LinkedHashMap<String, String>  validationErrors) {
		if( hasValue(value) && ! Validations.isValidDecimal( value ) ) {
			validationErrors.put(fieldName, "Invalid Number for " + fieldName);
			//messages.addMessage(MessageUtils.createErrorMessage(CoreMessages.INVALID_NUMBER, new String [] {property}, new Object[]{ value }));
		}
	}
	
	public static void validateDate(String fieldName, String month, String day, String year, boolean dayRequired, LinkedHashMap<String, String>  validationErrors) {
		if (!Validations.isValidDate(month, day, year, dayRequired)) {
			validationErrors.put(fieldName, "Invalid Date for " + fieldName);
			//messages.addMessage(MessageUtils.createErrorMessage(CoreMessages.INVALID_DATE, new String [] {property}, new Object[]{}));
		}
	}
	
	public static void validateTimestamp(String fieldName, String month, String day, String year, String hours, String minutes, boolean dayRequired, LinkedHashMap<String, String>  validationErrors) {
		if (!Validations.isValidTimestamp(month, day, year, hours, minutes, dayRequired)) {
			validationErrors.put(fieldName, "Invalid Date for " + fieldName);
			//messages.addMessage(MessageUtils.createErrorMessage(CoreMessages.INVALID_DATE, new String [] {property}, new Object[]{}));
		}
	}

	private static boolean isValidNumber(String s) {
		try {
			BigDecimal dec = new BigDecimal(s);
			return dec != null;
		} catch (Exception ex) {
			return false;
		}
	}
	
	/**
	 * Answers whether a string can be converted into an integer
	 * @param value
	 * @return
	 */
	private static boolean isValidInteger(String value) {
		try {
			Integer.parseInt( value );
			return true;
		}
		catch(NumberFormatException nfe) {
			return false;
		}
	}

	
	private static boolean isValidDate(String month, String day, String year, boolean dayRequired) {
		// A blank date may or may not be valid - up the the BO
		if (!hasValue(day) || !hasValue(month) && !hasValue(year)) {
			return true;
		}
		
		try {
			int iMonth = Integer.parseInt(month) - 1;
			int iDay;
			// Only default the day if it is not required AND blank.
			// In any other case try to parse it
			if (!dayRequired && !hasValue(day)) {
				iDay = 1;
			} else {
				iDay = Integer.parseInt(day);
			}
			int iYear = Integer.parseInt(year);
			
			Calendar cal = GregorianCalendar.getInstance();
			cal.clear();
			cal.setLenient(false);
			cal.set(Calendar.MONTH, iMonth);
			cal.set(Calendar.DAY_OF_MONTH, iDay);
			cal.set(Calendar.YEAR, iYear);
			
			// This will cause an exception if the day is invalid (e.g. 2/29/2001)
			cal.getTime();
			
			return true;
		} catch(Exception ex) {
			return false;
		}
	}
	
	private static boolean isValidTimestamp(String month, String day, String year, String hours, String minutes, boolean dayRequired) {
		try {
			boolean validDate = isValidDate(month, day, year, dayRequired);
			// TODO Handle time portion
			return validDate;
		} catch(Exception pe) {
			return false;
		}
	}

	/**
	 * Anserws whether a string can be converted into a long
	 * @param value
	 * @return
	 */
	
	private static boolean isValidLong(String value) {
		try {
			Long.parseLong( value );
			return true;
		}
		catch(NumberFormatException nfe) {
			return false;
		}
	}

	
	/**
	 * Anserws whether a string can be converted into a decimal
	 * @param value
	 * @return
	 */
	private static boolean isValidDecimal(String value) {
		try {
			new BigDecimal( value );
			return true;
		}
		catch(NumberFormatException nfe) {
			return false;
		}
	}

}
